#include <stdlib.h>
#include <string.h>

#include "global_types.h"

#include "control/job_scheduler.h"
#include "control/kmodule_fsm.h"
#include "control/worker_thread_set.h"

#include "model/kmodule_list.h"

#include "util/logger.h"

typedef enum
{
	SCHEDULING_RESULT_BUSY,
	SCHEDULING_RESULT_MODULES_QUEUED,
	SCHEDULING_RESULT_SCHEDULING_ONGOING
} scheduling_result_t;

typedef struct
{
	struct kmodule_list_t *module_list;
} job_scheduler_context_t;

job_scheduler_context_t job_scheduler_context={NULL};

static error_code_t job_scheduler_process_one_module(kmodule_t *kmodule, scheduling_result_t *sched_result);
static error_code_t job_scheduler_schedule_dep_module(kmodule_t *waiting_module, struct kmod_module *kmod);
static bool job_scheduler_check_dep_module_scheduled(kmodule_t *waiting_module, struct kmod_module *kmod);

//----------------------------------------------- job scheduler api members ----------------------------------------------
error_code_t job_scheduler_init(void)
{
	error_code_t result=RESULT_OK;

	job_scheduler_context.module_list=kmodule_list_create();
	if (job_scheduler_context.module_list==NULL)
		result=RESULT_NORESOURCES;

	logger_log_debug("JOB_SCHEDULER -> init done (result: %d).",result);
	return RESULT_OK;
}

void job_scheduler_deinit(void)
{
	if (job_scheduler_context.module_list!=NULL)
		kmodule_list_free(job_scheduler_context.module_list);
	logger_log_debug("JOB_SCHEDULER -> deinit done");
}

error_code_t job_scheduler_schedule_module(const char *module_name_or_path, 
		const char *module_params)
{
	error_code_t result;
	kmodule_t *kmodule;
	kmodule_list_entry_t *list_entry_ptr;

	//in case we find a .ko at the end of the string, we assume that we got a path not a name
	if (strstr(module_name_or_path,".ko")!=NULL)
		result=kmodule_new_from_path(&kmodule,module_name_or_path,module_params);
	else
		result=kmodule_new_from_name(&kmodule,module_name_or_path,module_params);

	if (result==RESULT_MODULE_LOADED)
	{
		logger_log_info("JOB_SCHEDULER -> Module %s already loaded. Skipping it.",module_name_or_path);
		result=RESULT_OK;
	}
	else if (result==RESULT_MODULE_BUILTIN)
	{
		logger_log_info("JOB_SCHEDULER -> Module %s is built in. Skipping it.",module_name_or_path);
		result=RESULT_OK;
	}
	else if (result==RESULT_MODULE_NOT_FOUND)
	{
		logger_log_error("Module %s not found.",module_name_or_path);
	}

	// in case of an error or if the module is built in or already loaded, The given pointer
	// is set to NULL by kmodule_new_from_xxx
	if (kmodule!=NULL)
	{
		logger_log_info("JOB_SCHEDULER -> Add module to list: %s", module_name_or_path);
		kmodule_list_queue_module(job_scheduler_context.module_list,kmodule,&list_entry_ptr);
		kmodule_set_mainlist_entry(kmodule,list_entry_ptr);
	}

	return result;
}

error_code_t job_scheduler_process_modules(void)
{
	kmodule_t *kmodule;
	kmodule_list_iterator_t itr;
	error_code_t result=RESULT_OK;
	scheduling_result_t sched_result=SCHEDULING_RESULT_SCHEDULING_ONGOING;

	if (job_scheduler_is_done())
		return RESULT_OK;

	kmodule=kmodule_list_first_entry(job_scheduler_context.module_list, &itr);

	while (sched_result == SCHEDULING_RESULT_SCHEDULING_ONGOING)
	{
		result=job_scheduler_process_one_module(kmodule, &sched_result);
		if (result!=RESULT_OK)
			return result;

		if (sched_result==SCHEDULING_RESULT_SCHEDULING_ONGOING)
		{
			//seems we couldn't do anything with the module. Take the next one
			kmodule=kmodule_list_next_entry(&itr);
			if (kmodule==NULL)
				//no module found to go on with something -> we are busy working on modules
				sched_result=SCHEDULING_RESULT_BUSY;
		}
		else if (sched_result==SCHEDULING_RESULT_MODULES_QUEUED)
		{
			//start finding modules for scheduling from the beginning of the list in case new modules have been inserted
			kmodule=kmodule_list_first_entry(job_scheduler_context.module_list, &itr);
			sched_result=SCHEDULING_RESULT_SCHEDULING_ONGOING;
		}
	}

	return result;
}

bool job_scheduler_is_done(void)
{
	return kmodule_list_is_empty(job_scheduler_context.module_list);
}

error_code_t job_scheduler_schedule_depmods(kmodule_t *kmodule, int *depmod_scheduled_cnt_ptr)
{
	error_code_t result;
	struct kmod_list *depmod_list;
	struct kmod_list *list_entry;
	struct kmod_module *kmod;
	int deps_scheduled_cnt;

	deps_scheduled_cnt=0;
	result=RESULT_OK;

	//check for dependencies and schedule them if needed
	depmod_list=kmod_module_get_dependencies(kmodule_get_kmod(kmodule));
	kmod_list_foreach(list_entry, depmod_list)
	{
		kmod=kmod_module_get_module(list_entry);
		if (job_scheduler_check_dep_module_scheduled(kmodule,kmod))
		{
			//we found it in our scheduler list so it has not been processed. In this case, we are waiting for it
			deps_scheduled_cnt++;
			continue;
		}

		//module not in the list, we add it to the scheduler
		result=job_scheduler_schedule_dep_module(kmodule,kmod);
		if (result==RESULT_OK)
			deps_scheduled_cnt++;

		kmod_module_unref(kmod);

		if (result==RESULT_MODULE_BUILTIN || result == RESULT_MODULE_LOADED)
			result=RESULT_OK;

		if (result!=RESULT_OK)
			break;
	}
	kmod_module_unref_list(depmod_list);

	if (depmod_scheduled_cnt_ptr!=NULL && result==RESULT_OK)
		*depmod_scheduled_cnt_ptr=deps_scheduled_cnt;

	return result;
}
//-------------------------------------------------------------------------------------------------------------------------

//----------------------------------------------- job scheduler private members -------------------------------------------
static error_code_t job_scheduler_process_one_module(kmodule_t *kmodule, scheduling_result_t *sched_result)
{
	error_code_t result;
	kmodule_state_t cur_mod_state;
	kmodule_state_t new_mod_state;
	kmodule_t *first_in_lst;
	kmodule_t *now_first_in_list;

	logger_log_debug("JOB_SCHEDULER -> scheduler looks at module %s.",kmodule_get_name(kmodule));

	cur_mod_state=kmodule_get_state(kmodule);

	//don't try to schedule something if the module is not in state ADDED or READY_TO_SCHEDULE
	if (cur_mod_state!=ADDED && cur_mod_state!=READY_TO_SCHEDULE)
	{
		logger_log_debug("JOB_SCHEDULER -> module neither in state ADDED nor in state READY_TO_SCHEDULE. Leaving it alone.");

		//tell the scheduler we can't do anything with this model. It will pick the next in the list.
		*sched_result=SCHEDULING_RESULT_SCHEDULING_ONGOING;
		return RESULT_OK;
	}

	first_in_lst=kmodule_list_first_entry(job_scheduler_context.module_list,NULL);

	result=kmodule_fsm_signal_schedule_job(kmodule);
	new_mod_state=kmodule_get_state(kmodule);

	now_first_in_list=kmodule_list_first_entry(job_scheduler_context.module_list,NULL);

	if (result==RESULT_OK)
	{
		if (cur_mod_state==ADDED && new_mod_state==DEPMOD_SCHEDULED)
		{
			if (first_in_lst != now_first_in_list)
				//we queued modules, let the scheduler start at the beginning of the list again
				(*sched_result)=SCHEDULING_RESULT_MODULES_QUEUED;
			else
				//we are waiting for dependencies, let the scheduler go on with the next module
				(*sched_result)=SCHEDULING_RESULT_SCHEDULING_ONGOING;
		}

		if (new_mod_state==READY_TO_SCHEDULE)
			//we tried to assign the module to one of the threads but we failed and stayed in (or got into)
			//READY_TO_SCHEDULE state. This happens when all threads are busy by now
			(*sched_result)=SCHEDULING_RESULT_BUSY;
	}

	return result;
}

static bool job_scheduler_check_dep_module_scheduled(kmodule_t *waiting_module, struct kmod_module *kmod)
{
	kmodule_t *module;
	module=kmodule_list_get_entry(job_scheduler_context.module_list,kmod);

	if (module!=NULL)
	{
		//add the module waitung for the new one to its list of waiting modules
		kmodule_add_waiting_module(module, waiting_module);

		logger_log_debug("KMODULE_FSM -> Module %s already scheduled.", kmodule_get_name(module));

		kmod_module_unref(kmod);
		//module already in list
		return true;
	}

	return false;
}

static error_code_t job_scheduler_schedule_dep_module(kmodule_t *waiting_module, struct kmod_module *kmod)
{
	error_code_t result;
	kmodule_t *new_kmodule=NULL;
	kmodule_list_entry_t *list_entry_ptr=NULL;

	result=kmodule_new_from_mod(&new_kmodule, kmod,NULL);

	logger_log_info("JOB_SCHEDULER -> Scheduling module %s to resolve dependencies.",kmod_module_get_name(kmod));

	if (result==RESULT_MODULE_BUILTIN || result==RESULT_MODULE_LOADED)
		logger_log_info("Module built in or loaded. Skipping it.");

	//queue the new module
	if (result==RESULT_OK)
		result=kmodule_list_queue_module(job_scheduler_context.module_list,new_kmodule,&list_entry_ptr);

	if (result==RESULT_OK)
	{
		//link the list entry into the kmodule structure
		kmodule_set_mainlist_entry(new_kmodule, list_entry_ptr);

		//add the module waitung for the new one to its list of waiting modules
		kmodule_add_waiting_module(new_kmodule, waiting_module);
	}

	//don't need to clean up if something went wrong. We are exiting in this case anyway.

	return result;
}
//-------------------------------------------------------------------------------------------------------------------------
